import Image from "next/image"
import Link from "next/link"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { ExternalLink, ArrowLeft } from "lucide-react"
import { notFound } from "next/navigation"
import { getToolBySlug } from "@/lib/supabase/tools"
import { createClient } from "@/lib/supabase/server"
import MarkdownContent from "@/components/markdown-content"

interface ToolPageProps {
  params: {
    slug: string
  }
}

export default async function ToolPage({ params }: ToolPageProps) {
  // Get the tool directly from Supabase
  const supabase = await createClient();
  const { data: tool, error } = await supabase
    .from('ai_tools')
    .select('*, categories(name)')
    .eq('slug', params.slug)
    .eq('is_approved', true)
    .single();
  
  if (error || !tool) {
    notFound();
  }

  return (
    <div className="container mx-auto px-4 py-8">
      <Link href="/" className="inline-flex items-center text-sm mb-6">
        <ArrowLeft className="mr-2 h-4 w-4" />
        返回首页
      </Link>

      {/* 顶部封面图 */}
      <div className="relative w-full h-64 rounded-lg overflow-hidden mb-8">
        <Image
          src={
            tool.image_url || "/placeholder.svg?height=400&width=1200&query=abstract digital pattern for hero image"
          }
          alt={tool.name}
          fill
          className="object-cover"
          priority
        />
      </div>

      <div className="max-w-3xl mx-auto">
        <div className="flex items-center gap-4 mb-6">
          <div className="relative h-16 w-16 shrink-0 overflow-hidden rounded-md">
            <Image src={tool.logo_url || "/placeholder.svg"} alt={tool.name} fill className="object-cover" />
          </div>
          <div>
            <h1 className="text-3xl font-bold">{tool.name}</h1>
            <div className="flex items-center gap-2 mt-2">
              <Badge>{tool.categories?.name}</Badge>
            </div>
          </div>
        </div>

        <Button className="mb-8" asChild>
          <a href={tool.website_url} target="_blank" rel="noopener noreferrer" className="inline-flex items-center">
            打开网站
            <ExternalLink className="ml-2 h-4 w-4" />
          </a>
        </Button>

        <div className="prose max-w-none">
          <div className="mb-6">
            <h2 className="text-xl font-bold mb-2">描述</h2>
            <p>{tool.description}</p>
          </div>
          
          {tool.full_description && (
            <div className="mt-6">
              <h2 className="text-xl font-bold mb-2">详细介绍</h2>
              <MarkdownContent content={tool.full_description} />
            </div>
          )}
        </div>
      </div>
    </div>
  )
}
